package com.example.mediademo.util;

import android.media.AudioTrack;
import android.os.Handler;
import android.os.Message;
import android.util.Log;


/***
 * Description : 使用AudioTrack播放音乐
 */
public class AudioPlayer {
    private static final String TAG = "AudioPlayer";
    public static final int STATE_MSG_ID = 16;
    private Handler mHandler;
    //音频参数
    private AudioParamBean mAudioParam;
    //音频数据
    private byte[] mData;
    //AudioTrack对象
    private AudioTrack mAudioTrack;
    //播放源是否就绪
    private boolean mBReady = false;
    //播放线程
    private PlayAudioThread mPlayAudioThread;
    //线程退出标志
    private boolean mThreadExitFlag = false;
    //较优播放块大小
    private int mPrimePlaySize = 0;
    //当前播放位置
    private int mPlayOffset = 0;
    //当前播放状态, 0 未初始化, 2 正在播放
    private int mPlayState = 0;

    public AudioPlayer(Handler handler) {
        mHandler = handler;
    }

    //设置播放参数
    public void setAudioParam(AudioParamBean audioParam) {
        mAudioParam = audioParam;
    }

    //设置播放源
    public void setDataSource(byte[] data) {
        mData = data;
    }    

    //准备播放源
    public boolean prepare() {
        if (mData != null && mAudioParam != null) {
            if (mBReady) {
                return true;
            } else {
                try {
                    createAudioTrack();
                } catch (Exception var2) {
                    var2.printStackTrace();
                    return false;
                }

                mBReady = true;
                setPlayState(1);
                return true;
            }
        } else {
            return false;
        }
    }   

    public boolean release() {
        stop();
        releaseAudioTrack();
        mBReady = false;
        setPlayState(0);
        return true;
    }

    public boolean start() {
        if (!mBReady) {
            return false;
        } else {
            switch (mPlayState) {
                case 1:
                    mPlayOffset = 0;
                    setPlayState(2);
                    startThread();
                case 2:
                default:
                    break;
                case 3:
                    setPlayState(2);
                    startThread();
            }

            return true;
        }
    }

    public boolean pause() {
        if (!mBReady) {
            return false;
        } else {
            if (mPlayState == 2) {
                setPlayState(3);
                stopThread();
            }

            return true;
        }
    }

    public boolean stop() {
        if (!mBReady) {
            return false;
        } else {
            setPlayState(1);
            stopThread();
            return true;
        }
    }

    public boolean isPlaying(){
        return mPlayState==2;
    }

    private synchronized void setPlayState(int state) {
        mPlayState = state;
        if (mHandler != null) {
            Message msg = mHandler.obtainMessage(16);
            msg.obj = mPlayState;
            msg.sendToTarget();
        }

    }

    /*
     *构建一个AudioTrack对象
     */
    private void createAudioTrack() throws Exception {
        int minBufSize = AudioTrack.getMinBufferSize(mAudioParam.mSampleRateInHZ, mAudioParam.mChannel, mAudioParam.mAudioFormat);
        mPrimePlaySize = minBufSize * 2;
        Log.d("AudioPlayer", "mPrimePlaySize = " + mPrimePlaySize);
        mAudioTrack = new AudioTrack(3, mAudioParam.mSampleRateInHZ, mAudioParam.mChannel, mAudioParam.mAudioFormat, minBufSize, 1);
    }

    private void releaseAudioTrack() {
        if (mAudioTrack != null) {
            mAudioTrack.stop();
            mAudioTrack.release();
            mAudioTrack = null;
        }

    }

    private void startThread() {
        if (mPlayAudioThread == null) {
            mThreadExitFlag = false;
            mPlayAudioThread = new PlayAudioThread();
            mPlayAudioThread.start();
        }

    }

    private void stopThread() {
        if (mPlayAudioThread != null) {
            mThreadExitFlag = true;
            mPlayAudioThread = null;
        }

    }

    public void onPlayComplete() {
        mPlayAudioThread = null;
        if (mPlayState != 3) {
            setPlayState(1);
        }

    }

    /*
     *播放音频线程
     */
    class PlayAudioThread extends Thread {
        PlayAudioThread() {
        }

        public void run() {
            mAudioTrack.play();

            while (!mThreadExitFlag) {
                try {
                    mAudioTrack.write(mData, mPlayOffset, mPrimePlaySize);
                    mPlayOffset = mPlayOffset + mPrimePlaySize;
                } catch (Exception var2) {
                    var2.printStackTrace();
                    onPlayComplete();
                    break;
                }

                // 循环播放
                if (mPlayOffset >= mData.length) {
                    mPlayOffset = 0;
                }
            }

            mAudioTrack.stop();
            Log.d("AudioPlayer", "PlayAudioThread complete...");
        }

    }
}
